
#region References

using System;
using System.Data;
using System.Data.SqlClient;
using System.Reflection;
using gov.va.med.vbecs.Common;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;

#endregion

namespace gov.va.med.vbecs.DAL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Greg Lohse</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>04/29/2004</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	/// PatientTransfusionReaction data access layer class.
	///</summary>

	#endregion
	
	public class PatientTransfusionReaction
	{
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/21/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4453"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable schema</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4454"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates a DataTable of the PatientTransfusionReaction table
		/// </summary>
		/// <returns>Empty DataTable</returns>
		public static DataTable GetEmptyPatientTransfusionReactionTable()
		{
			System.Data.DataTable dt = new System.Data.DataTable(TABLE.PatientTransfusionReaction.TableName);

			dt.Columns.Add(TABLE.PatientTransfusionReaction.DivisionCode, typeof(string));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.EntryDateTime, typeof(System.DateTime));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.EntryTechId, typeof(string));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.InvestigatedDateTime, typeof(System.DateTime));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.LastUpdateDate, typeof(System.DateTime));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.LastUpdateUser, typeof(string));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.MedicalDirectorName, typeof(string));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.NotedDateTime, typeof(System.DateTime));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.OrderedTestGuid, typeof(System.Guid));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.PartialSaveId, typeof(byte));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.PatientGuid, typeof(System.Guid));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.PatientTransfusionReactionGuid, typeof(System.Guid));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.PostSpecimenClericalChecksOkIndicator, typeof(bool));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.PreSpecimenClericalChecksOkIndicator, typeof(bool));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.PostSpecimenNotAvailableIndicator, typeof(bool));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.PreSpecimenNotAvailableIndicator, typeof(bool));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.PreSpecimenHemolysisInspectionCode, typeof(char));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.PostSpecimenHemolysisInspectionCode, typeof(char));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.RecordStatusCode, typeof(string));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.RowVersion, typeof(byte []));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.TransfusionReactionTypeCode, typeof(string));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.PreSpecimenGuid, typeof(System.Guid));
			dt.Columns.Add(TABLE.PatientTransfusionReaction.PostSpecimenGuid, typeof(System.Guid));

			return(dt);
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/14/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4696"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4697"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets full list of all hemolysis inspection codes and descriptions
		/// </summary>
		/// <returns>DataTable of hemolysis inspection codes and descriptions</returns>
		public static DataTable GetHemolysisInspectionList()
		{
			return Common.StoredProcedure.GetData(STOREDPROC.GetHemolysisInspectionList.StoredProcName).Tables[0];
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/17/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4652"> 
		///		<ExpectedInput>PatientGuid, DivisionCode</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4653"> 
		///		<ExpectedInput>Invalid GUid</ExpectedInput>
		///		<ExpectedOutput>Argument Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets blood units implicated in transfusion reaction
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <param name="divisionCode"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetImplicatedBloodUnits(Guid patientGuid, string divisionCode)
		{
			if (patientGuid.Equals(Guid.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient unique identifier").ResString);
			}
			//
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("division code").ResString);
			}
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetTransfusionReactionImplicatedUnitsForPatient.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetTransfusionReactionImplicatedUnitsForPatient.divisioncode, System.Data.SqlDbType.Char)
			};
			//
			prms[0].Value = patientGuid;
			prms[1].Value = divisionCode;
			//
			return Common.StoredProcedure.GetData(STOREDPROC.GetTransfusionReactionImplicatedUnitsForPatient.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/17/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2165"> 
		///		<ExpectedInput>PatientGuid divisioncode</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2168"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>Argument Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets transfusion records implicated in transfusion reaction
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <param name="divisionCode"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetImplicatedTransfusionRecords(Guid patientGuid, string divisionCode)
		{
			if (patientGuid.Equals(Guid.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient unique identifier").ResString);
			}
			//
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("division code").ResString);
			}
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetPatientTransfusionRecords.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetPatientTransfusionRecords.divisioncode, System.Data.SqlDbType.Char)
			};
			//
			prms[0].Value = patientGuid;
			prms[1].Value = divisionCode;
			//
			return Common.StoredProcedure.GetData(STOREDPROC.GetPatientTransfusionRecords.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/17/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2169"> 
		///		<ExpectedInput>PatientTransfusionReactionGuid divisioncode</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2174"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>Argument Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets transfusion records implicated in transfusion reaction
		/// </summary>
		/// <param name="patientTransfusionReactionGuid"></param>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetImplicatedTransfusionRecordsForPatientTransfusionReaction(Guid patientTransfusionReactionGuid, string divisionCode)
		{
			if (patientTransfusionReactionGuid.Equals(Guid.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient transfusion reaction unique identifier").ResString);
			}
			//
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("division code").ResString);
			}
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetImplicatedUnitsForPatientTransfusionReaction.patienttransfusionreactionguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetImplicatedUnitsForPatientTransfusionReaction.divisioncode, System.Data.SqlDbType.Char)
			};
			//
			prms[0].Value = patientTransfusionReactionGuid;
			prms[1].Value = divisionCode;
			//
			return Common.StoredProcedure.GetData(STOREDPROC.GetImplicatedUnitsForPatientTransfusionReaction.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/17/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4632"> 
		///		<ExpectedInput>OrderedTestGuid, DivisionCode</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4633"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>Argument Exceptions</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets patient transfusion reaction record data
		/// </summary>
		/// <param name="orderedTestGuid"></param>
		/// <param name="divisionCode"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetPatientTransfusionReaction(Guid orderedTestGuid, string divisionCode)
		{
			if (orderedTestGuid.Equals(Guid.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient transfusion reaction identifier").ResString);
			}
			//
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("division code").ResString);
			}
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetPatientTransfusionReaction.orderedtestguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetPatientTransfusionReaction.divisioncode, System.Data.SqlDbType.Char)
			};
			//
			prms[0].Value = orderedTestGuid;
			prms[1].Value = divisionCode;
			//
			return Common.StoredProcedure.GetData(STOREDPROC.GetPatientTransfusionReaction.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/25/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8100"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8101"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetCompletedPatientTransfusionReactions
		/// </summary>
		/// <returns></returns>
		public static DataTable GetCompletedPatientTransfusionReactions()
		{
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetCompletedPatientTransfusionReactions.divisioncode, System.Data.SqlDbType.Char)
			};
			//
			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			//
			return Common.StoredProcedure.GetData(STOREDPROC.GetCompletedPatientTransfusionReactions.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/25/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8102"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>DataSet</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8103"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataSet</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetPatientTransfusionReactionReport
		/// </summary>
		/// <param name="patientTransfusionReactionGuid"></param>
		/// <returns></returns>
		public static DataSet GetPatientTransfusionReactionReport(Guid patientTransfusionReactionGuid)
		{
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetReportPatientTransfusionReaction.patienttransfusionreactionguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetReportPatientTransfusionReaction.divisioncode, System.Data.SqlDbType.Char)
			};
			//
			prms[0].Value = patientTransfusionReactionGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;
			//
			return Common.StoredProcedure.GetData(STOREDPROC.GetReportPatientTransfusionReaction.StoredProcName, prms);
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4650"> 
		///		<ExpectedInput>OrderedTestGuid, division</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4651"> 
		///		<ExpectedInput>invalid parameters</ExpectedInput>
		///		<ExpectedOutput>argument exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets ABO/Rh and XM Blood Unit Test data
		/// </summary>
		/// <param name="orderedTestGuid"></param>
		/// <param name="divisionCode"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetSerologicBloodUnitTests(Guid orderedTestGuid, string divisionCode)
		{
			if (orderedTestGuid.Equals(Guid.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("ordered test unique identifier").ResString);
			}
			//
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("division code").ResString);
			}
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetBoodUnitTestResultsForTransfusionReaction.orderedtestguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetBoodUnitTestResultsForTransfusionReaction.divisioncode, System.Data.SqlDbType.Char)
			};
			//
			prms[0].Value = orderedTestGuid;
			prms[1].Value = divisionCode;
			//
			return Common.StoredProcedure.GetData(STOREDPROC.GetBoodUnitTestResultsForTransfusionReaction.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4638"> 
		///		<ExpectedInput>OrderedTestGuid, division</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4639"> 
		///		<ExpectedInput>invalid parameters</ExpectedInput>
		///		<ExpectedOutput>argument exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets ABO/Rh, ABS and DAT Specimen Test data
		/// for both pre and post transfusion specimens.
		/// </summary>
		/// <param name="orderedTestGuid"></param>
		/// <param name="divisionCode"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetSerologicSpecimenTests(Guid orderedTestGuid, string divisionCode)
		{
			if (orderedTestGuid.Equals(Guid.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("ordered test unique identifier").ResString);
			}
			//
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("division code").ResString);
			}
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetSpecimenTestResultsForTransfusionReaction.orderedtestguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetSpecimenTestResultsForTransfusionReaction.divisioncode, System.Data.SqlDbType.Char)
			};
			//
			prms[0].Value = orderedTestGuid;
			prms[1].Value = divisionCode;
			//
			return Common.StoredProcedure.GetData(STOREDPROC.GetSpecimenTestResultsForTransfusionReaction.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4640"> 
		///		<ExpectedInput>PatientTransfusionReactionGuid, division</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4641"> 
		///		<ExpectedInput>invalid parameters</ExpectedInput>
		///		<ExpectedOutput>argument exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves symptoms data for a particular
		/// patient transfusion reaction
		/// </summary>
		/// <param name="patientTransfusionReactionGuid"></param>
		///  <param name="divisionCode"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetSymtoms(Guid patientTransfusionReactionGuid, string divisionCode)
		{
			if (patientTransfusionReactionGuid.Equals(Guid.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient transfusion reaction unique identifier").ResString);
			}
			//
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("division code").ResString);
			}
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetPatientTransfusionReactionSymptoms.patienttransfusionreactionguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetPatientTransfusionReactionSymptoms.divisioncode, System.Data.SqlDbType.Char)
			};
			//
			prms[0].Value = patientTransfusionReactionGuid;
			prms[1].Value = divisionCode;
			//
			return Common.StoredProcedure.GetData(STOREDPROC.GetPatientTransfusionReactionSymptoms.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/14/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4698"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4699"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get complete list of all possible symptoms
		/// </summary>
		/// <returns>DataTable of symptom ids and descriptions</returns>
		public static DataTable GetPatientSymptomList()
		{
			return Common.StoredProcedure.GetData(STOREDPROC.GetPatientSymptomList.StoredProcName).Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/28/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4159"> 
		///		<ExpectedInput>Valid PatientGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4160"> 
		///		<ExpectedInput>Invalid PatientGuid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetTransfusionReactionsForPatient
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static DataTable GetTransfusionReactionsForPatient(System.Guid patientGuid)
		{
			SqlParameter[] prms = 
				{
					new SqlParameter(STOREDPROC.GetTransfusionReactionsForPatient.patientguid, SqlDbType.UniqueIdentifier),
					new SqlParameter(STOREDPROC.GetTransfusionReactionsForPatient.divisioncode, SqlDbType.Char),
				};

			prms[0].Value = patientGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetTransfusionReactionsForPatient.StoredProcName, prms);
			
			return (ds.Tables[0]);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8187"> 
		///		<ExpectedInput>Valid patient guid</ExpectedInput>
		///		<ExpectedOutput>Count of transfusion reactions</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8188"> 
		///		<ExpectedInput>Invalid patient guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		///	Get count of transfusion reactions for a patient
		/// </summary>
		/// <param name="patientGuid">Patient Guid</param>
		/// <returns>Transfusion reaction count</returns>
		public static int GetTransfusionReactionsForPatientCount(System.Guid patientGuid)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patientGuid").ResString);
			}

			int count = 0;
			//
			SqlParameter[] prms = 
				{
					new SqlParameter(STOREDPROC.GetTransfusionReactionsForPatientCount.patientguid, SqlDbType.UniqueIdentifier),
					new SqlParameter(STOREDPROC.GetTransfusionReactionsForPatientCount.divisioncode, SqlDbType.Char),
			};

			prms[0].Value = patientGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataTable dt = Common.StoredProcedure.GetData(STOREDPROC.GetTransfusionReactionsForPatientCount.StoredProcName, prms).Tables[0];
			
			foreach(DataRow dr in dt.Rows)
			{
				count += (int)dr["Total"];
			}
			//
			return count;
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/17/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="5011"> 
		///		<ExpectedInput>PatientTransfusionReactionGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5012"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Argument Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets transfusion reaction records implicated in transfusion reaction
		/// </summary>
		/// <param name="patientTransfusionReactionGuid"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetTransfusionReactionRecords(Guid patientTransfusionReactionGuid)
		{
			if (patientTransfusionReactionGuid.Equals(Guid.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient transfusion reaction unique identifier").ResString);
			}
			//
			SqlParameter[] prms =
			{
				new SqlParameter( STOREDPROC.GetTransfusionReactionRecords.patienttransfusionreactionguid, System.Data.SqlDbType.UniqueIdentifier )
			};
			//
			prms[0].Value = patientTransfusionReactionGuid;
			//
			return Common.StoredProcedure.GetData( STOREDPROC.GetTransfusionReactionRecords.StoredProcName, prms ).Tables[0];
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/5/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4642"> 
		///		<ExpectedInput>PatientTransfusionReactionGuid, RowVersion</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4643"> 
		///		<ExpectedInput>invalid RowVersion</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8141"> 
		///		<ExpectedInput>Valid Guid</ExpectedInput>
		///		<ExpectedOutput>Argument Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Sets RecordStatus code to 'X' (inactive) for a TRW
		/// </summary>
		/// <param name="patientTransfusionReactionGuid"></param>
		/// <param name="orderedTest">Ordered Test table</param>
		/// <param name="rowVersion"></param>
		/// <returns>Success Indicator</returns>
		public static bool Invalidate(Guid patientTransfusionReactionGuid, DataTable orderedTest, byte [] rowVersion)
		{
			if (patientTransfusionReactionGuid.Equals(Guid.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient transfusion reaction unique identifier").ResString);
			}

			DataTable patientTransfusionReaction = new DataTable(TABLE.PatientTransfusionReaction.TableName);
			patientTransfusionReaction.Columns.Add(TABLE.PatientTransfusionReaction.PatientTransfusionReactionGuid, typeof(Guid));
			patientTransfusionReaction.Columns.Add(TABLE.PatientTransfusionReaction.DivisionCode, typeof(string));
			patientTransfusionReaction.Columns.Add(TABLE.PatientTransfusionReaction.RowVersion, typeof(byte[]));
			//
			DataRow dr = patientTransfusionReaction.NewRow();
			dr[TABLE.PatientTransfusionReaction.PatientTransfusionReactionGuid] = patientTransfusionReactionGuid;
			dr[TABLE.PatientTransfusionReaction.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			dr[TABLE.PatientTransfusionReaction.RowVersion] = rowVersion;
			patientTransfusionReaction.Rows.Add(dr);
			//
			System.Collections.ArrayList dtArray = new System.Collections.ArrayList();
			System.Collections.ArrayList spArray = new System.Collections.ArrayList();
			//
			spArray.Add(STOREDPROC.InvalidateTransfusionReactionWorkup.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(patientTransfusionReaction, Common.UpdateFunction.UC021_FrmTransfusionReactionWorkup));
			//
			spArray.Add(STOREDPROC.UpdateOrderedTestTaskStatus.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(orderedTest, Common.UpdateFunction.UC021_FrmTransfusionReactionWorkup));
			//
			return ((new StoredProcedure()).TransactionalGetValue(spArray, dtArray) == 0);
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/5/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2177"> 
		///		<ExpectedInput>DataTables</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2180"> 
		///		<ExpectedInput>invalid RowVersions</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Invalidate
		/// </summary>
		/// <param name="patientTransfusionReaction"></param>
		/// <param name="bloodUnitTestResults"></param>
		/// <param name="specimenTestResults"></param>
		/// <param name="orderedTest"></param>
		/// <param name="lastUpdateFunctionId"></param>
		/// <param name="dtWorkloadEvent"></param>
		/// <returns></returns>
		public static bool Invalidate(DataTable patientTransfusionReaction, DataTable bloodUnitTestResults, DataTable specimenTestResults, DataTable orderedTest, DataTable dtWorkloadEvent, Common.UpdateFunction lastUpdateFunctionId)
		{
			System.Collections.ArrayList dtArray = new System.Collections.ArrayList();
			System.Collections.ArrayList spArray = new System.Collections.ArrayList();
			//
			spArray.Add(STOREDPROC.InvalidateTransfusionReactionWorkup.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(patientTransfusionReaction, lastUpdateFunctionId));
			//
			spArray.Add(STOREDPROC.UpdateSpecimenTestStatus.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(specimenTestResults, lastUpdateFunctionId));
			//
			spArray.Add(STOREDPROC.UpdateBloodUnitTestRecordStatus.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(bloodUnitTestResults, lastUpdateFunctionId));
			//
			if(dtWorkloadEvent != null)
			{
				if(dtWorkloadEvent.Rows.Count > 0)
				{
					dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtWorkloadEvent, lastUpdateFunctionId));
					spArray.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);
				}
			}

			if (orderedTest != null)
			{
				if(orderedTest.Rows.Count>0)
				{
					dtArray.Add(Common.Utility.AppendLastUpdateInformation(orderedTest, lastUpdateFunctionId));
					spArray.Add(STOREDPROC.UpdateOrderedTestStatus.StoredProcName);
				}
			}
			return ((new Common.StoredProcedure()).TransactionalGetValue(spArray, dtArray) == 0);

		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/5/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4644"> 
		///		<ExpectedInput>DataTables</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4645"> 
		///		<ExpectedInput>Invalid Table Schemas</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8142"> 
		///		<ExpectedInput>Empty patient transfusion reaction table</ExpectedInput>
		///		<ExpectedOutput>Argument exception thrown</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Save patient transfusion and comment data for blood
		/// units implicated in a reaction.  The unit/transfusion
		/// data are implicated via an entry in TransfusionReaction 
		/// and there may exist many such entries (up to one per
		/// transfused unit).
		/// </summary>
		/// <param name="patientTransfusionReaction"></param>
		/// <param name="transfusionReaction"></param>
		/// <param name="patientTransfusionComment"></param>
		/// <param name="trwExceptions"></param>
		/// <param name="lastUpdateFunctionId"></param>
		/// <returns>DataTable</returns>
		public static bool SaveImplicatedBloodUnits(DataTable patientTransfusionReaction, DataTable transfusionReaction, DataTable patientTransfusionComment, DataTable trwExceptions, UpdateFunction lastUpdateFunctionId)
		{
			if (patientTransfusionReaction.Rows.Count==0)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument("patient transfusion reaction").ResString,"patient transfusion reaction");
			}
			//
			Common.Utility.RequireNonNullColumns(patientTransfusionReaction, STOREDPROC.UpdatePatientTransfusionReactionPartialSaveId.RequiredParameters);
			Common.Utility.RequireNonNullColumns(transfusionReaction, STOREDPROC.InsertTransfusionReaction.RequiredParameters);
			//
			System.Collections.ArrayList dtArray = new System.Collections.ArrayList();
			System.Collections.ArrayList spArray = new System.Collections.ArrayList();
			//
			spArray.Add(STOREDPROC.UpdatePatientTransfusionReactionPartialSaveId.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(patientTransfusionReaction, lastUpdateFunctionId));
			//
			spArray.Add(STOREDPROC.InsertTransfusionReaction.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(transfusionReaction, lastUpdateFunctionId));
			//
			spArray.Add(STOREDPROC.InsertPatientTransfusionComment.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(patientTransfusionComment, lastUpdateFunctionId));
			//
			if (trwExceptions != null)
			{
				spArray.Add( STOREDPROC.InsertExceptionTransfusionReactionWorkup.StoredProcName );
				dtArray.Add(trwExceptions);
			}
			//
			return ((new Common.StoredProcedure()).TransactionalGetValue(spArray, dtArray) == 0);
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/5/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4648"> 
		///		<ExpectedInput>DataTables</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4649"> 
		///		<ExpectedInput>Invalid schemas</ExpectedInput>
		///		<ExpectedOutput>ArgumentExceptions</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Saves blood unit and specimen tests
		/// </summary>
		/// <param name="patientTransfusionReaction"></param>
		/// <param name="specimenTest"></param>
		/// <param name="bloodUnitTest"></param>
		/// <param name="orderedTest"></param>
		/// <param name="exceptions"></param>
		/// <param name="lastUpdateFunctionId"></param>
		/// <param name="workload"></param>
		/// <returns>DataTable</returns>
		public static bool SaveSerologicTests(DataTable patientTransfusionReaction, DataTable specimenTest, DataTable bloodUnitTest, DataTable orderedTest, DataTable exceptions, UpdateFunction lastUpdateFunctionId, DataTable workload)
		{
			if (patientTransfusionReaction.Rows.Count==0)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument("patient transfusion reaction").ResString,"patient transfusion reaction");
			}
			//
			Common.Utility.RequireNonNullColumns(patientTransfusionReaction, STOREDPROC.UpdatePatientTransfusionReactionPartialSaveId.RequiredParameters);
			Common.Utility.RequireNonNullColumns(specimenTest, STOREDPROC.InsertSpecimenTest.RequiredParameters);
			Common.Utility.RequireNonNullColumns(bloodUnitTest, STOREDPROC.InsertBloodUnitTest.RequiredParameters);
			Common.Utility.RequireNonNullColumns(orderedTest, STOREDPROC.UpdateOrderedTestTaskStatus.RequiredParameters);
			Common.Utility.RequireNonNullColumns(workload, STOREDPROC.InsertWorkloadEvents.RequiredParameters);
			//
			System.Collections.ArrayList dtArray = new System.Collections.ArrayList();
			System.Collections.ArrayList spArray = new System.Collections.ArrayList();
			//
			spArray.Add(STOREDPROC.UpdatePatientTransfusionReactionPartialSaveId.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(patientTransfusionReaction, lastUpdateFunctionId));
			//
			spArray.Add(STOREDPROC.InsertSpecimenTest.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(specimenTest, lastUpdateFunctionId));
			//
			spArray.Add(STOREDPROC.InsertBloodUnitTest.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(bloodUnitTest, lastUpdateFunctionId));
			//
			spArray.Add(STOREDPROC.UpdateOrderedTestTaskStatus.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(orderedTest, lastUpdateFunctionId));
			//
			if (exceptions != null)
			{
				spArray.Add(STOREDPROC.InsertExceptionQCNotPerformed.StoredProcName);
				dtArray.Add(exceptions);
			}

			if(workload != null)
			{
				spArray.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);
				dtArray.Add(Common.Utility.AppendLastUpdateInformation(workload, lastUpdateFunctionId));
			}
			//
			return ((new StoredProcedure()).TransactionalGetValue(spArray, dtArray) == 0);
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/5/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4634"> 
		///		<ExpectedInput>DataTables</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4635"> 
		///		<ExpectedInput>Invalid schemas</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8143"> 
		///		<ExpectedInput>Empty data table</ExpectedInput>
		///		<ExpectedOutput>Argument Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Saves specimen check info for a TRW.  
		/// Note: this includes inserting the 
		/// PatientTransfusionReaction row.
		/// </summary>
		/// <param name="patientTransfusionReaction"></param>
		/// <param name="orderedTest"></param>
		/// <param name="patientTransfusionComment"></param>
		/// <param name="exceptions"></param>
		/// <param name="lastUpdateFunctionId"></param>
		/// <returns>DataTable</returns>
		public static bool SaveSpecimenChecks(DataTable patientTransfusionReaction, DataTable orderedTest, DataTable patientTransfusionComment, DataTable exceptions, UpdateFunction lastUpdateFunctionId)
		{
			if (patientTransfusionReaction.Rows.Count==0)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument("patient transfusion reaction").ResString,"patient transfusion reaction");
			}
			//
			Common.Utility.RequireNonNullColumns(patientTransfusionReaction, STOREDPROC.InsertPatientTransfusionReaction.RequiredParameters);
			Common.Utility.RequireNonNullColumns(orderedTest, STOREDPROC.UpdateOrderedTestTaskStatus.RequiredParameters);
			//
			System.Collections.ArrayList dtArray = new System.Collections.ArrayList();
			System.Collections.ArrayList spArray = new System.Collections.ArrayList();
			//
			spArray.Add(STOREDPROC.InsertPatientTransfusionReaction.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(patientTransfusionReaction, lastUpdateFunctionId));
			//
			spArray.Add(STOREDPROC.UpdateOrderedTestTaskStatus.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(orderedTest, lastUpdateFunctionId));
			//
			spArray.Add(STOREDPROC.InsertPatientTransfusionComment.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(patientTransfusionComment, lastUpdateFunctionId));
			//
			if (exceptions != null)
			{
				spArray.Add( STOREDPROC.InsertExceptionTrwSpecimen.StoredProcName );
				dtArray.Add( exceptions );
			}
			//
			return ((new Common.StoredProcedure()).TransactionalGetValue(spArray, dtArray) == 0);
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/5/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4646"> 
		///		<ExpectedInput>DataTables</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4647"> 
		///		<ExpectedInput>Invalid schemas</ExpectedInput>
		///		<ExpectedOutput>ArgumentExceptions</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8144"> 
		///		<ExpectedInput>Empty data table</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Saves symptoms and "other detail" for a TRW
		/// </summary>
		/// <param name="patientTransfusionReaction"></param>
		/// <param name="patientTransfusionReactionSymptom"></param>
		/// <param name="patientTransfusionComment"></param>
		/// <param name="lastUpdateFunctionId"></param>
		/// <returns>DataTable</returns>
		public static bool SaveSymptons(DataTable patientTransfusionReaction, DataTable patientTransfusionReactionSymptom, DataTable patientTransfusionComment, UpdateFunction lastUpdateFunctionId)
		{
			if (patientTransfusionReaction.Rows.Count==0)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument("patient transfusion reaction").ResString,"patient transfusion reaction");
			}
			//
			Common.Utility.RequireNonNullColumns(patientTransfusionReactionSymptom, STOREDPROC.InsertPatientTransfusionReactionSymptom.RequiredParameters);
			//
			System.Collections.ArrayList dtArray = new System.Collections.ArrayList();
			System.Collections.ArrayList spArray = new System.Collections.ArrayList();
			//
			spArray.Add(STOREDPROC.UpdatePatientTransfusionReactionPartialSaveId.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(patientTransfusionReaction, lastUpdateFunctionId));
			//
			spArray.Add(STOREDPROC.InsertPatientTransfusionReactionSymptom.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(patientTransfusionReactionSymptom, lastUpdateFunctionId));
			//
			if (patientTransfusionComment != null)
			{
				spArray.Add(STOREDPROC.InsertPatientTransfusionComment.StoredProcName);
				dtArray.Add(Common.Utility.AppendLastUpdateInformation(patientTransfusionComment, lastUpdateFunctionId));
			}
			//
			return ((new StoredProcedure()).TransactionalGetValue(spArray, dtArray) == 0);
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/14/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4700"> 
		///		<ExpectedInput>DataTables</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4701"> 
		///		<ExpectedInput>Invalid Schema</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8145"> 
		///		<ExpectedInput>Empty data table</ExpectedInput>
		///		<ExpectedOutput>Argument Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates a TRW's record status code 
		/// </summary>
		/// <param name="patientTransfusionReaction"></param>
		/// <param name="lastUpdateFunctionId"></param>
		/// <returns>success indicator</returns>
		public static bool UpdateStatus(DataTable patientTransfusionReaction, UpdateFunction lastUpdateFunctionId)
		{
			if (patientTransfusionReaction.Rows.Count==0)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument("patient transfusion reaction").ResString,"patient transfusion reaction");
			}
			//
			Common.Utility.RequireNonNullColumns(patientTransfusionReaction, STOREDPROC.UpdatePatientTransfusionReactionStatus.RequiredParameters);
			//
			return ((new StoredProcedure()).TransactionalGetValue(STOREDPROC.UpdatePatientTransfusionReactionStatus.StoredProcName, Common.Utility.AppendLastUpdateInformation(patientTransfusionReaction, lastUpdateFunctionId)) == 0);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/25/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8104"> 
		///		<ExpectedInput>Valid parameters</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8105"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// FinalizeReport
		/// </summary>
		/// <param name="patientTransfusionReaction"></param>
		/// <param name="patientTransfusionComment"></param>
		/// <param name="lastUpdateFunctionId"></param>
		/// <returns></returns>
		public static bool FinalizeReport(DataTable patientTransfusionReaction, DataTable patientTransfusionComment, UpdateFunction lastUpdateFunctionId)
		{
			System.Collections.ArrayList dtArray = new System.Collections.ArrayList();
			System.Collections.ArrayList spArray = new System.Collections.ArrayList();
			//
			spArray.Add(STOREDPROC.FinalizePatientTransfusionReaction.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(patientTransfusionReaction, lastUpdateFunctionId));
			//
			spArray.Add(STOREDPROC.InsertPatientTransfusionComment.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(patientTransfusionComment, lastUpdateFunctionId));
			//
			return ((new StoredProcedure()).TransactionalGetValue(spArray, dtArray) == 0);

		}
	}
}
